#
# * GEOSYNCHRONOUS CONSTELLATIONS
# *
# * Mars stationary constellation
# *
# * Similar to the Clarke three-satellite geostationary Earth system,
# * but at different longitudes to minimise drift and stationkeeping. See:
# *
# * Stationkeeping in Mars Orbit, Emily Lakdawalla, The Planetary Society,
# * 27 June 2013.
# https://www.planetary.org/articles/stationkeeping-in-mars-orbit
# *
# * Optimal longitudes determination for the station keeping
# * of areostationary satellites, J. J. Silva and P. Romero,
# * Planetary and Space Science, vol. 87, October 2013, pp. 14-18.
# https://doi.org/10.1016/j.pss.2012.11.013
#
# this script for SaVi by Lloyd Wood (lloydwood@users.sourceforge.net)
#
# $Id: mars-stationary.tcl 265 2024-10-05 10:43:20Z lloydwood $

# It's easiest to give each satellite its own RAAN, so we'll treat them
# as having separate planes even though they're not actually ascending.
set NUM_PLANES 3

# set location of one satellite; the others will be spaced from it
# east is positive.
# Mars stable points are 17.92W and 167.83E, so we choose one of those
# then space equally. The other two satellites will have more drift.
set sat_longitude 167.83

# setup orbital elements

# match the current stationary altitude.
# see tcl/equator.tcl for working
set OM $params(Omega)

set cuberoot [expr 1.0/3]
set a [expr { 0.0 + pow(($MU / $OM / $OM), $cuberoot) } ]

set radius [format "%0.2f" $a]
puts stderr "\nSaVi: stationary radius is $radius km."

set e 0.0
set inc 0.0
set omega 0.0
set Omega 0.0
set T_per [expr 2 * $PI * pow($a,1.5) / sqrt($MU)]

# for illustration purposes, but also a rough guide to practical use.
set coverage_angle 5.0

satellites GV_BEGIN

# equatorial, so turn on equatorial plane
upvar #0 plane_flag plane_flag
set plane_flag 1

for {set i 0} {$i < $NUM_PLANES} {incr i} {
   # SaVi handles wraparound beyond +/-180
   set Omega [ expr 360.0/$NUM_PLANES * $i + $sat_longitude]
   set T 0
   satellites LOAD $a $e $inc $Omega $omega $T "Mars-stationary-$i"
}

satellites GV_END
