#
######################################################
#
#  SaVi by Lloyd Wood (lloydwood@users.sourceforge.net),
#          Patrick Worfolk (worfolk@alum.mit.edu) and
#          Robert Thurman.
#
#  Copyright (c) 1997 by The Geometry Center.
#  Also Copyright (c) 2017 by Lloyd Wood.
#
#  This file is part of SaVi.  SaVi is free software;
#  you can redistribute it and/or modify it only under
#  the terms given in the file COPYRIGHT which you should
#  have received along with this file.  SaVi may be
#  obtained from:
#  http://savi.sourceforge.net/
#  http://www.geom.uiuc.edu/locate/SaVi
#
######################################################
#
# params.tcl
#
# $Id: params.tcl 267 2024-10-06 06:21:20Z lloydwood $


proc params(build) {} {
    global PI tparams params_display

    if {[eval window(raise) params]} return

    set name [build_Toplevel params]

    wm protocol $name WM_DELETE_WINDOW params(dismiss)

    build_Title $name "Simulation constants"

    set cmd [build_StdFrame $name cmd]

    build_IPopupMenu $cmd p1 "" params(Body) params(central_body) { \
	"Earth" "Mars"}
    
    pack $cmd.p1 -side top -expand 1

    set cmda [build_StdFrame $cmd a]
    
    params(push_to_display)

    build_LabelEntryColumns $cmda le0 \
	    {text " " {"Radius (km)" "Rotation period (s)"} } \
	    {dentry "Central body" { tparams(Radius) params_display(rotation_period_s) } }

    pack $cmda -side left

    set cmdb [build_StdFrame $cmd b]

    build_LabelEntryColumns $cmdb le1 \
	{text " " {"Mu (km^3/s^2)" "Oblateness (J2)"} } \
	{dentry "Gravitational model" { tparams(Mu) tparams(J2) } }

    pack $cmdb -side left

    set cmdc [build_StdFrame $cmd c]
    
    build_LabelEntryColumns $cmdc le1 \
	{text " " {"year (Earth days)" "Axial tilt (degrees)"} } \
	{dentry "Orbit around Sun" { params_display(orbital_period_days) tparams(tilt) } }

    pack $cmdc -expand 1
    
    pack $cmd -expand 1


    build_DismissButtonbar $name dbbar "params(dismiss)" \
	{"Reset to default" params(reset)} \
	{"Cancel" params(cancel)} \
	{"Apply" params(apply)}


    update

    # fill in the boxes
    params(cancel)
}

proc params(dismiss) {} {
    destroy .params
    params(cancel)
}

proc params(push_to_display) {} {
    global PI tparams params_display

    # params(orbital_period) is around the Sun.
    # shown in hours, as easier to understand.
    # days would depend on rotation period or Earth days, best avoided.
    set params_display(orbital_period_days) [expr $tparams(orbital_period) / 86400]
    
    # We show length of day in seconds as it is easier to understand.
    # this wouldn't work for Venus, which rotates counter.
    set params_display(rotation_period_s) [expr 2.0*$PI/$tparams(Omega)]
}

proc params(push_from_tparams) {} {
    global params tparams
    
    set inds [array names tparams]
    foreach ind $inds {
        set params($ind) $tparams($ind)
    }
}

proc params(copy_to_tparams) {} {
    global params tparams
    
    set inds [array names params]
    foreach ind $inds {
        set tparams($ind) $params($ind)
    }
}

proc params(apply) {} {
    global PI params tparams params_display latitude_line

    # negative body radius and negative square roots are best avoided.
    if {$tparams(Radius) < 0} {
	set tparams(Radius) -$tparams(Radius)
    }

    if {$tparams(Mu) < 0} {
	set tparams(Mu) -$tparams(Mu)
    }

    if {$params_display(rotation_period_s) < 0} {
	set params_display(rotation_period_s) -$params_display(rotation_period_s)
    }

    if {$params_display(orbital_period_days) < 0} {
	set params_display(orbital_period_days) -$params_display(orbital_period_days)
    }


    # add 0.0 to force floating point, just in case.
    set params_display(orbital_period_days) [expr 0.0 + $params_display(orbital_period_days)]
    set params_display(rotation_period_s) [expr 0.0 + $params_display(rotation_period_s)]
	
    set tparams(J2) [expr 0.0 + $tparams(J2)]
    set tparams(Mu) [expr 0.0 + $tparams(Mu)]
    set tparams(tilt) [expr 0.0 + $tparams(tilt)]
    
    # check inclination to ecliptic plane is in limits.
    if {$tparams(tilt) < -90} {
	set tparams(tilt) -90.0
    } elseif {$tparams(tilt) > 90} {
        set tparams(tilt) 90.0
    }

    # params is definitive, we compute from display
    set tparams(Omega) [expr 2.0*$PI/$params_display(rotation_period_s)]
    set tparams(orbital_period) [expr $params_display(orbital_period_days) * 3600 * 24]

    params(push_from_tparams)
    
    # pass tilt inclination through to C for tropics
    # already used in sunlight inclination
    if {$latitude_line != $params(tilt)} {
        set latitude_line $params(tilt)
	coverage(update)
    }

    # now update other panels if necessary
    edit(update)
}

proc params(cancel) {} {

    params(copy_to_tparams)

    params(push_to_display)
}

proc params(reset) {} {

    params(central_body)
}

proc params(Earth) {} {
    global PI params

    set params(Body) 0
    
    set params(Radius) 6378.14

    # Earth sidereal day 86164.0905
    set params(Omega) [expr 2 * $PI / 86164.0905]
    set params(tilt) 23.5
    set params(Mu) 398600.4418
    set params(J2) 0.0010826

    # approximation used for sun calculation. Adjusts for leap years.
    set params(orbital_period) [expr 365.25 * 24 * 3600]

    params(copy_to_tparams)
}

proc params(Mars) {} {
    global PI params

    set params(Body) 1
    
    set params(Radius) 3396.2

    # Mars sidereal day 88,642.663 seconds
    set params(Omega) [expr 2 * $PI / 88642.663]
    set params(tilt) 25.19
    set params(Mu) 42828.37
    set params(J2) 0.00196045
    # also need MARS_OBLATENESS

    # approximation used for sun calculation. 687 Earth days.
    set params(orbital_period) [expr 687.0 * 24 * 3600 ]

    params(copy_to_tparams)
}


proc params(central_body) {} {
    global PI params main_constellations_submenu coverage_constellations_submenu
    global fisheye_constellations_submenu fisheye_locations_submenu map_flag

    # forcing map flag off for non-Earth is crude,
    # doesn't address land/sea
    if {$params(Body) == 0} {
	params(Earth)
	set map_flag 1
    } else {
	params(Mars)
	set map_flag 0
    }

    params(push_to_display)
    
    params(apply)

    if [winfo exists .main] {
	main(constellations_menu) $main_constellations_submenu
    }

    coverage(restore_colors)
    if [winfo exists .coverage] {
	coverage(constellations_menu) $coverage_constellations_submenu
    }

    if [winfo exists .fisheye] {
	fisheye(constellations_menu) $fisheye_constellations_submenu
	fisheye(locations_menu) $fisheye_locations_submenu
    }
}

proc params(init) {} {

    #Earth is default
    params(Earth)

    params(push_from_tparams)
}
